<?php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, OPTIONS');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Get server URL dynamically
$protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
$host = $_SERVER['HTTP_HOST'];
$server = "$protocol://$host";

// Main parameters
$path = $_GET['path'] ?? '';
$action = $_GET['action'] ?? 'list';

// Base directories
$filesDir  = __DIR__ . '/files';
$thumbsDir = __DIR__ . '/thumbs';
if (!is_dir($thumbsDir)) mkdir($thumbsDir, 0777, true);

switch($action) {
    case 'list':
        listFiles($path);
        break;
    case 'delete':
        deleteFile($path);
        break;
    default:
        echo json_encode(['status'=>'ERR','msg'=>'Unknown action']);
        break;
}

// ---- Functions ----

function listFiles($pathQuery) {
    global $filesDir, $thumbsDir, $server;

    $scanDir = $filesDir . ($pathQuery ? "/$pathQuery" : '');
    $items = [];

    // Totals
    $rootStats = getFolderStatsRecursive($filesDir);
    $currentStats = ['files'=>0,'folders'=>0,'size'=>0,'size_formatted'=>'0 B'];

    if (is_dir($scanDir)) {
        $currentStats = getFolderStatsRecursive($scanDir);
        $currentStats['size_formatted'] = formatFileSize($currentStats['size']);

        foreach (scandir($scanDir) as $f) {
            if ($f === '.' || $f === '..') continue;
            $fullPath = "$scanDir/$f";
            $relPath = $pathQuery ? "$pathQuery/$f" : $f;

            if (is_dir($fullPath)) {
                $stats = getFolderStatsRecursive($fullPath);
                $items[] = [
                    'name' => $f,
                    'type' => 'folder',
                    'path' => $pathQuery,
                    'size' => $stats['size'],
                    'size_formatted' => formatFileSize($stats['size']),
                    'file_count' => $stats['files'],
                    'folder_count' => $stats['folders'],
                    'upload_date' => date('Y-m-d H:i:s', filemtime($fullPath))
                ];
            } else {
                $ext = strtolower(pathinfo($f, PATHINFO_EXTENSION));
                $isImage = in_array($ext, ['jpg','jpeg','png','gif','webp']);
                $thumbnail_url = null;

                if ($isImage) {
                    $thumbPath = $thumbsDir . ($pathQuery ? "/$pathQuery/$f" : "/$f");
                    if (!is_dir(dirname($thumbPath))) @mkdir(dirname($thumbPath), 0777, true);
                    if (!file_exists($thumbPath)) generateThumbnail($fullPath, $thumbPath);
                    if (file_exists($thumbPath)) {
                        // FIX: Remove the double server prefix - just use relative path
                        $thumbnail_url = "/thumbs/" . rawurlencode($relPath);
                    }
                }

                // Generate correct URLs
                $download_url = "$server/download.php?file=" . rawurlencode($relPath);
                $preview_url = "$server/files/" . rawurlencode($relPath);
                
                $items[] = [
                    'name' => $f,
                    'type' => 'file',
                    'path' => $pathQuery,
                    'size' => filesize($fullPath),
                    'size_formatted' => formatFileSize(filesize($fullPath)),
                    'upload_date' => date('Y-m-d H:i:s', filemtime($fullPath)),
                    'download_url' => $download_url,
                    'preview_url' => $preview_url,
                    'thumbnail_url' => $thumbnail_url
                ];
            }
        }
    }

    // Sort: folders first, newest files
    usort($items, function($a,$b){
        if($a['type']==='folder' && $b['type']!=='folder') return -1;
        if($a['type']!=='folder' && $b['type']==='folder') return 1;
        return strcmp($b['upload_date'],$a['upload_date']);
    });

    $response = [
        'items' => $items,
        'root_totals' => [
            'files' => $rootStats['files'],
            'folders' => $rootStats['folders'],
            'size' => $rootStats['size'],
            'size_formatted' => formatFileSize($rootStats['size'])
        ],
        'current_totals' => $currentStats
    ];

    echo json_encode($response);
}

function deleteFile($pathQuery) {
    global $filesDir, $thumbsDir;
    $target = realpath($filesDir . '/' . $pathQuery);

    if ($target && strpos($target, realpath($filesDir)) === 0) {
        if (is_dir($target)) {
            rrmdir($target);
            $thumbTarget = $thumbsDir . '/' . $pathQuery;
            if (is_dir($thumbTarget)) rrmdir($thumbTarget);
        } else {
            unlink($target);
            $thumbFile = $thumbsDir . '/' . $pathQuery;
            if (file_exists($thumbFile)) @unlink($thumbFile);
        }
        echo json_encode(['status'=>'OK']);
    } else {
        echo json_encode(['status'=>'ERR','msg'=>'bad path']);
    }
}

// Helpers (keep the same as before)
function formatFileSize($bytes){
    if ($bytes>=1073741824) return number_format($bytes/1073741824,2).' GB';
    if ($bytes>=1048576) return number_format($bytes/1048576,2).' MB';
    if ($bytes>=1024) return number_format($bytes/1024,2).' KB';
    return $bytes.' bytes';
}

function getFolderStatsRecursive($path){
    $totalFiles=0; $totalFolders=0; $totalSize=0;
    foreach(scandir($path) as $f){
        if($f==='.' || $f==='..') continue;
        $full="$path/$f";
        if(is_dir($full)){
            $totalFolders++;
            $sub=getFolderStatsRecursive($full);
            $totalFiles+=$sub['files'];
            $totalFolders+=$sub['folders'];
            $totalSize+=$sub['size'];
        } else {
            $totalFiles++;
            $totalSize+=filesize($full);
        }
    }
    return ['files'=>$totalFiles,'folders'=>$totalFolders,'size'=>$totalSize];
}

function rrmdir($dir){
    foreach(glob($dir.'/*') as $f){
        is_dir($f)? rrmdir($f): unlink($f);
    }
    rmdir($dir);
}

function generateThumbnail($src, $dest){
    if (!file_exists($src)) return false;
    $ext = strtolower(pathinfo($src, PATHINFO_EXTENSION));
    if (!in_array($ext,['jpg','jpeg','png','gif','webp'])) return false;

    if (!is_dir(dirname($dest))) @mkdir(dirname($dest),0777,true);

    $info = @getimagesize($src);
    if (!$info) return false;
    list($w,$h,$type) = $info;

    $max=150;
    $ratio = $w>0 && $h>0?($w/$h):1;
    if($w>$h){ $nw=$max; $nh=intval($max/$ratio); } else { $nh=$max; $nw=intval($max*$ratio); }

    switch($type){
        case IMAGETYPE_JPEG: $img=@imagecreatefromjpeg($src); break;
        case IMAGETYPE_PNG: $img=@imagecreatefrompng($src); break;
        case IMAGETYPE_GIF: $img=@imagecreatefromgif($src); break;
        case IMAGETYPE_WEBP: $img=@imagecreatefromwebp($src); break;
        default: return false;
    }
    if (!$img) return false;

    $thumb = imagecreatetruecolor($nw,$nh);
    if($type==IMAGETYPE_PNG||$type==IMAGETYPE_GIF){
        imagealphablending($thumb,false);
        imagesavealpha($thumb,true);
        $transparent = imagecolorallocatealpha($thumb,255,255,255,127);
        imagefilledrectangle($thumb,0,0,$nw,$nh,$transparent);
    }

    imagecopyresampled($thumb,$img,0,0,0,0,$nw,$nh,$w,$h);

    $saved=false;
    switch($type){
        case IMAGETYPE_JPEG: $saved=imagejpeg($thumb,$dest,85); break;
        case IMAGETYPE_PNG: $saved=imagepng($thumb,$dest,8); break;
        case IMAGETYPE_GIF: $saved=imagegif($thumb,$dest); break;
        case IMAGETYPE_WEBP: $saved=imagewebp($thumb,$dest,85); break;
    }
    imagedestroy($img);
    imagedestroy($thumb);
    return $saved;
}
?>