<?php
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

try {
    $input = json_decode(file_get_contents('php://input'), true);
    
    $uploadId = $input['uploadId'] ?? '';
    $path = $input['path'] ?? 'edge/' . date('Y-m');
    $fileName = $input['fileName'] ?? 'file';
    $decompress = $input['decompress'] ?? false;
    
    error_log("🔗 CHUNK_ASSEMBLE - Starting: ID: $uploadId, Path: $path, File: $fileName, Decompress: " . ($decompress ? 'yes' : 'no'));

    // Sanitize
    $path = 'files/' . trim($path, '/');
    $path = preg_replace('/\.\./', '', $path);
    
    $chunkDir = "{$path}/chunks/{$uploadId}";
    $finalPath = "{$path}/{$fileName}";
    
    error_log("📁 CHUNK_ASSEMBLE - Chunk dir: $chunkDir");
    error_log("📁 CHUNK_ASSEMBLE - Final path: $finalPath");
    
    if (!is_dir($chunkDir)) {
        throw new Exception('No chunks found for upload ID: ' . $uploadId);
    }
    
    // Get and sort chunks numerically
    $chunks = glob("{$chunkDir}/*.chunk");
    if (empty($chunks)) {
        throw new Exception('No chunk files found');
    }
    
    error_log("📦 CHUNK_ASSEMBLE - Found " . count($chunks) . " chunks");
    
    // Sort chunks by their numeric index
    usort($chunks, function($a, $b) {
        $aIndex = intval(basename($a, '.chunk'));
        $bIndex = intval(basename($b, '.chunk'));
        return $aIndex - $bIndex;
    });
    
    // Assemble file with optional decompression
    $final = fopen($finalPath, 'wb');
    
    if (!$final) {
        throw new Exception('Failed to create final file: ' . $finalPath);
    }
    
    $processedChunks = 0;
    foreach ($chunks as $chunk) {
        $chunkData = file_get_contents($chunk);
        
        if ($decompress) {
            // Decompress gzipped chunk data
            $decompressed = @gzdecode($chunkData);
            
            if ($decompressed === false) {
                // If gzdecode fails, try alternative decompression methods
                $decompressed = @gzinflate(substr($chunkData, 10)); // Skip gzip header
                
                if ($decompressed === false) {
                    throw new Exception('Failed to decompress chunk: ' . basename($chunk));
                }
            }
            
            fwrite($final, $decompressed);
        } else {
            // Regular chunk - write as-is
            fwrite($final, $chunkData);
        }
        
        // Clean up chunk file
        if (unlink($chunk)) {
            error_log("✅ CHUNK_ASSEMBLE - Deleted chunk: " . basename($chunk));
            $processedChunks++;
        } else {
            error_log("⚠️ CHUNK_ASSEMBLE - Failed to delete chunk: " . basename($chunk));
        }
    }
    
    fclose($final);
    
    // Verify final file was created
    if (!file_exists($finalPath) || filesize($finalPath) === 0) {
        throw new Exception('Final file assembly failed - file is empty or missing');
    }
    
    error_log("🎉 CHUNK_ASSEMBLE - Assembly complete, processed $processedChunks chunks");
    
    // 🧹 ENHANCED CLEANUP - Remove all empty directories
    cleanupAllDirectories($chunkDir);
    
    //$finalUrl = 'https://o.kunok.com/' . $finalPath;
    // TO THIS:
$protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
$host = $_SERVER['HTTP_HOST'];
echo "{$protocol}://{$host}/" . $finalPath;
    error_log("✅ CHUNK_ASSEMBLE - Final URL: $finalUrl");
    
    echo $finalUrl;
    
} catch (Exception $e) {
    error_log("❌ CHUNK_ASSEMBLE - Error: " . $e->getMessage());
    http_response_code(500);
    echo 'Error: ' . $e->getMessage();
}

// ENHANCED CLEANUP - Recursively remove empty directories
function cleanupAllDirectories($startDir) {
    if (!is_dir($startDir)) {
        error_log("⚠️ CLEANUP - Directory doesn't exist: $startDir");
        return;
    }
    
    error_log("🧹 CLEANUP - Starting cleanup from: $startDir");
    
    // Remove the specific upload ID directory
    if (is_dir($startDir)) {
        // First remove all files
        $files = glob($startDir . '/*');
        foreach ($files as $file) {
            if (is_file($file)) {
                unlink($file);
            }
        }
        
        // Then remove the directory
        if (rmdir($startDir)) {
            error_log("✅ CLEANUP - Removed upload directory: $startDir");
        } else {
            error_log("⚠️ CLEANUP - Failed to remove upload directory: $startDir");
        }
    }
    
    // Now work upwards and remove empty parent directories
    $currentDir = dirname($startDir); // chunks folder for this path
    $maxLevels = 3; // Prevent infinite loops
    
    for ($i = 0; $i < $maxLevels; $i++) {
        if (!is_dir($currentDir)) {
            break;
        }
        
        // Check if directory is empty
        $files = array_diff(scandir($currentDir), ['.', '..']);
        
        if (empty($files)) {
            // Directory is empty - remove it
            if (rmdir($currentDir)) {
                error_log("✅ CLEANUP - Removed empty directory: $currentDir");
                
                // Move up to parent directory
                $currentDir = dirname($currentDir);
            } else {
                error_log("⚠️ CLEANUP - Failed to remove directory: $currentDir");
                break;
            }
        } else {
            error_log("⚠️ CLEANUP - Directory not empty, keeping: $currentDir (contains: " . implode(', ', $files) . ")");
            break;
        }
    }
}

// Alternative: Simple force cleanup (if recursive doesn't work)
function forceCleanup($chunkDir) {
    if (!is_dir($chunkDir)) {
        return;
    }
    
    error_log("💥 FORCE CLEANUP - Removing: $chunkDir");
    
    // Remove all files in the chunk directory
    $files = glob($chunkDir . '/*');
    foreach ($files as $file) {
        if (is_file($file)) {
            unlink($file);
        }
    }
    
    // Remove the chunk directory
    if (rmdir($chunkDir)) {
        error_log("✅ FORCE CLEANUP - Removed: $chunkDir");
    }
    
    // Try to remove parent chunks directory if empty
    $parentDir = dirname($chunkDir);
    if (is_dir($parentDir) && basename($parentDir) === 'chunks') {
        $parentFiles = array_diff(scandir($parentDir), ['.', '..']);
        if (empty($parentFiles)) {
            if (rmdir($parentDir)) {
                error_log("✅ FORCE CLEANUP - Removed empty chunks directory: $parentDir");
            }
        }
    }
    
    // Try to remove grandparent directory if empty (the path directory)
    $grandParentDir = dirname($parentDir);
    if (is_dir($grandParentDir)) {
        $grandParentFiles = array_diff(scandir($grandParentDir), ['.', '..']);
        if (empty($grandParentFiles)) {
            if (rmdir($grandParentDir)) {
                error_log("✅ FORCE CLEANUP - Removed empty path directory: $grandParentDir");
            }
        }
    }
}
?>